//////////////////////////////////////////////////////////////////////////////
//  Copyright 2016 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.widget - Defines widgets objects
 */

var ntp = ntp || {};
ntp.widget = ntp.widget || {};

var showTooltip = function () {
    $(this).attr('title', $(this).attr('data-title'));
}

var hideTooltip = function () {
    $(this).attr('title', '');
}

/**
 * Get Start widget
 */
ntp.widget.getStarted = {
    options : {
        title : 'column_get_started_title'
    },

    _create : function() {
        // title
        ntp.widget.showWidget($('<div>'), 'Title').appendTo(this.element);

        // get started box
        ntp.widget.showWidget($('<div>'), 'GetStartedBox').appendTo(this.element);

        // template bar
        ntp.widget.showWidget($('<div>'), 'Templatebar').appendTo(this.element);

        // command groups
        ntp.widget.showWidget($('<div>'), 'CommandsGroup').appendTo(this.element);
    }
};

/**
 * Title Widget
 */

ntp.widget.title = {
    options : {
        title : 'column_get_started_title'
    },

    _create : function() {
        $('<h1>').attr('data-text-id', this.options.title).appendTo(this.element);
    }
};

/**
 * Tile widget (Get Started Box)
 */
ntp.widget.tile = {
    options : {
        caption : '',
        background : '',
        action : '',
        startDrawing: 'enabled'
    },

    _create : function() {
        if (this.options.startDrawing === 'enabled') {
            var $getStartedBox = $('<div>').attr({ 'id' : 'get_started_box'}).addClass('shadow_box');
            var $span = $('<span>').attr({ 'id' : 'start_drawing_span' }).appendTo($getStartedBox);

            // Caption
            if ( typeof this.options.caption == 'string' && this.options.caption !== '') {
                ntp.utils.localize($span, this.options.caption);
            } else {
                ntp.utils.localize($span, 'get_started_box_background');
            }

            // Background
            if ( typeof this.options.background === 'string' && this.options.background !== '') {
                // Mark $getStartedBox as custom
                $getStartedBox.addClass('custom');

                var backgroundUrl = "url('" + ntp.utils.resolvePath(this.options.background) + "')";

                ntp.utils.addRule('#get_started_box.custom {' + '   background-image: ' + backgroundUrl + ',' + '                      -webkit-linear-gradient(top, rgba(255,255,255,.1) 0%, rgba(240,240,240,.1) 30%, rgba(193,193,193,1) 100%);' + '}');
                ntp.utils.addRule('#get_started_box.custom:hover {' + '   background-image: ' + backgroundUrl + ',' + '                     -webkit-linear-gradient(top, rgba(255,255,255,0),rgba(0,156,248,.2));' + '}');
                ntp.utils.addRule('#get_started_box.custom:active {' + '   background-image: ' + backgroundUrl + ',' + '                     -webkit-linear-gradient(top, rgba(255,255,255,0),rgba(255,255,255,.2));' + '}');
            }

            // Action
            if (this.options.action === '')
                $getStartedBox.attr('onclick', 'ntp.acad.createDrawingFile()');
            else if ( typeof this.options.action === 'string')
                $getStartedBox.attr('onclick', '').off('click').click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', this.options.action));

            this.element.append($getStartedBox);
        } 
    }
};

/**
 * External widget which can load any widget which is defined in some file like *.html etc.
 *
 */
ntp.widget.external = {
    options : {
        url : ''    // required
    },
    _create : function() {
        var that = this.element;
        if (this.options.url !== '') {
            $.ajax({
                url : this.options.url,
                success : function(data) {
                    that.html(data);
                },
                async : false
            });
        }
    }
};

/**
 * template bar widget
 */
ntp.widget.templatebar = (function() {
    var $templatePanel = null;
    var $templateFooter = null;
    var toggling = false;
    var $listItemForlastUsedTemplate = null;

    // Defines the event handler that closes the panel when user click anywhere outside the panel
    var handleClickedOutsideTemplatePanel = function() {
        // Check if the mouse if hovering anywhere outside the panel
        if ((!$templatePanel.is(':hover') || $('#create_new_sheet_set').is(':hover')) && !toggling) {
            $templatePanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).off('click', handleClickedOutsideTemplatePanel);
        }
        toggling = false;
    };

    // Adjust the height of template panel
    var updateFloatingTemplatePanelHeight = function() {
        // Calculate the maximum height of the panel
        var $templateList = $('#template_list');
        var $templateBar = $('#template_bar');
        var $tabCreate = $('#tab_create');
        var templatePanelHeight = $tabCreate.offset().top + $tabCreate.height() - $templateBar.offset().top - $templateBar.outerHeight() - 24;
        var templateFooterHeight = 0;
        if ($templateFooter !== null) {
            templateFooterHeight = $templateFooter.outerHeight(true);
            if ($templatePanel.css('display') === 'none') {
                $templatePanel.css({ 'display' : 'block', 'height' : 0 });
                templateFooterHeight = $templateFooter.outerHeight(true);
                $templatePanel.css({ 'display' : 'none', 'height' : 'auto'});
            }
        }
        $templateList.css('max-height', templatePanelHeight - templateFooterHeight);
    };

    var processMenuActions = function(menuActions, parent) {
        if (!( menuActions instanceof Array)) {
            return;
        }

        // the id selector has some problem right now, let's use this way
        var $templateFooterGroups = parent.children().eq(1).children(1).eq(2).children().eq(0);

        $.each(menuActions, function(i, menuAction) {
            if (!( typeof menuAction.category === 'string' && typeof menuAction.text === 'string' && typeof menuAction.action === 'string')) {
                return;
            }

            var $group = $templateFooterGroups.children('[data-group="' + menuAction.category + '"]');
            if ($group.size() == 0) {
                // Create group
                $group = $('<li>').attr('data-group', menuAction.category);
                var $groupCaption = $('<span>');
                $groupCaption.appendTo($group);

                $group.append($('<ol>'));
                $group.appendTo($templateFooterGroups);

                ntp.utils.localize($groupCaption, menuAction.category);
            }

            // Create menu item
            var $menuItem = $('<li>').addClass('button');
            $group.children('ol').append($menuItem);

            ntp.utils.localize($menuItem, menuAction.text);

            if ( typeof menuAction.icon === 'string') {
                $menuItem.css('background-image', 'url("' + ntp.utils.resolvePath(menuAction.icon) + '")');
            }

            $menuItem.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', menuAction.action));
        });
    };

    var addNoTemplateToList = function() {
        var $noTempalteList = $('<li>');
        var $ol = $('<ol>');
        $ol.append($('<hr>'));
        // imperial
        $('<li>').attr('id', 'template_imperial').addClass('dwg button').text(ntp.utils.localize('no_template_imperial')).click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'createDrawingFileWithNoTemplate', true)).appendTo($ol);
        // metric
        $('<li>').attr('id', 'template_metric').addClass('dwg button').text(ntp.utils.localize('no_template_metric')).click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'createDrawingFileWithNoTemplate', false)).appendTo($ol);
        $noTempalteList.append($ol).appendTo($('#template_list'));
    };
    
    var loadTemplates = function(categories) {
        var i, j;
        for ( i = 0; i < categories.length; i++) {
            var catName = categories[i].name;
            var catFullPath = categories[i].fullPath;

            var $templateList = $('<li>');
            var $categoryNameSpan = null;
            var $arrowSpan = null;
            var $rightArrowSVG = null;
            var $downArrowSVG = null;
            var $itemDiv = null;

            if (catName !== '.') {
                $rightArrowSVG = $('<svg><polygon points="1,8 9,12 1,16" style="fill:#999"/></svg>').hide();
                $downArrowSVG = $('<svg><polygon points="1,8 9,8 5,16" style="fill:#999"/></svg>');
                $rightArrowSVG.attr('width', '10px').attr('height', '20px');
                $downArrowSVG.attr('width', '10px').attr('height', '20px');
                $arrowSpan = $('<span>').append($downArrowSVG).append($rightArrowSVG).css('display', 'block').css('margin-right', '3px');

                // Title attribute is used to show the tooltip info.
                $categoryNameSpan = $('<span>').text(catName).attr('data-title', catFullPath).css({ 'display' : 'block', 'width' : '232px' });
                $categoryNameSpan.hover(showTooltip, hideTooltip);

                $itemDiv = $('<div>').append($arrowSpan).append($categoryNameSpan);
                $itemDiv.css({ 'display' : '-webkit-box', '-webkit-box-align' : 'center', '-webkit-box-orient' : 'horizontal' });

                $templateList.append($itemDiv);
            }

            var $ol = $('<ol>').appendTo($templateList);

            var eventData = { 'ol' : $ol, 'rightArrowSVG' : $rightArrowSVG, 'downArrowSVG' : $downArrowSVG };

            if ($categoryNameSpan !== null)
                $categoryNameSpan.click(eventData, ntp.widget.templatebar.handleCategoryClicked);
            if ($arrowSpan != null)
                $arrowSpan.click(eventData, ntp.widget.templatebar.handleCategoryClicked);

            var files = categories[i].files;

            for (j = 0; j < files.length; j++) {

                var templateToolTipText ='';

                if (typeof files[j].description !== 'undefined' && files[j].description.length > 0)
                    templateToolTipText = files[j].description + '\r\n\n';

                templateToolTipText += files[j].fullPath;

                var $item = $('<li>').addClass('dwg button').text(files[j].name).attr('data-title', templateToolTipText);
                $item.hover(showTooltip, hideTooltip);

                $item.data({
                fullPath : files[j].fullPath
                });

                $ol.append($item);
                $item.click(ntp.widget.templatebar.handleItemClicked);
            }
            $('#template_list').append($templateList);
            
            if ( i === 0 )
                addNoTemplateToList();
        }
    };

    return {
        options : {
            createSheetSet : 'enabled',
            customMenuActions : [],
            templateBar: ''
        },

        _create : function() {
            if (this.options.templateBar !== "disabled") {
                var $div = ntp.widget.showWidgetWithOptions($('<div>'), 'External', {
                    url : 'js/widget_templates/templatebar.html'
                }).appendTo(this.element);

                if (this.options.createSheetSet !== 'enabled')
                    $div.children().eq(1).children(1).eq(2).children().eq(0).children().eq(0).remove();

                processMenuActions(this.options.customMenuActions, $div);

                this.beginRefresh();
            }
        },

        isEnabled : function() {
            return ntp.settings.templateBar !== 'disabled';
        },

        refresh : function(data) {
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            $('#template_list').empty();

            if ( typeof (data.templates) !== "undefined")
                loadTemplates(data.templates);

            // Toggle template empty placeholder.
            if ($('#template_list').children().length > 1) {
                $('#template_empty').addClass('hidden');
            } else {
                $('#template_empty').removeClass('hidden');
            }
        },

        beginRefresh : function() {
            if (!this.isEnabled()) {
                return null;
            }
            return ntp.acad.loadTemplateFilesAsJSON().done(ntp.utils.makeCallback(this, 'refreshWithJSON'));
        },
        
        hilightLastUsedTemplateListItem: function (data) {
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (!this.isEnabled() || typeof data === 'undefined' || typeof (data.templatename) === 'undefined')
                return;
            
            //Remove the previous hilight if any
            if ($listItemForlastUsedTemplate != null) {
                $listItemForlastUsedTemplate.css({ 'background': 'transparent', 'border-radius': '0' });
            }
            //Hilight the Item matching lastused template
            $('#template_list li.button').each(function () {
                if (typeof $(this).data('fullPath') !== 'undefined' && $(this).data('fullPath').toLocaleLowerCase() === data.templatename.toLocaleLowerCase()) {
                    $listItemForlastUsedTemplate = $(this);
                    $(this).css({ 'background': '-webkit-linear-gradient(top,#e5e5e5,#c8c8c8)', 'border-radius': '2px' });
                }
            });
        },

        refreshLastUsedTemplateWithJSON : function(lastTemplateFile){
            if (!this.isEnabled()) {
                return;
            }
            this.hilightLastUsedTemplateListItem($.parseJSON(lastTemplateFile));
        },

        refreshWithJSON : function(templatesJSON) {
            if (!this.isEnabled()) {
                return;
            }
            
            this.refresh($.parseJSON(templatesJSON));
            ntp.acad.getLastUsedTemplateAsJSON().done(ntp.utils.makeCallback(this, 'refreshLastUsedTemplateWithJSON'));
        },

        init : function() {
            if (ntp.settings.templateBar !== "disabled") {
                $templatePanel = $('#template_panel');
                var $footer = $('#template_footer');
                if ($footer.length > 0) {
                    $templateFooter = $footer;
                }
            }
        },

        update : function() {
            // Adjust the size of floating template panel
            if ($templatePanel !== null && $templatePanel.css('display') !== 'none') {
                updateFloatingTemplatePanelHeight();
            }
        },

        handleTemplateBarClicked : function(event) {
            if ($templatePanel.css('display') === 'none') {
                toggling = true;

                updateFloatingTemplatePanelHeight();

                // Show the template panel
                $templatePanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
                $(document).on('click', handleClickedOutsideTemplatePanel);
            }
        },

        // The browser button is only visible when there is not templates in the templates menu.
        handleTemplateBrowseButtonClicked : function(event) {
            ntp.acad.loadTemplateFilesFromSelectedFolder().done(function(templateFilesJSON) {
                ntp.widget.templatebar.refreshWithJSON(templateFilesJSON);
            });
        },

        handleItemClicked : function() {
            $templatePanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).off('click', handleClickedOutsideTemplatePanel);

            if(typeof $(this).data('fullPath') !== 'undefined')
                ntp.acad.createDrawingFileWithTemplate($(this).data('fullPath'));

            // Setting the title of the item to empty string to enforce turning off tooltip.
            $(this).attr('title', '');
        },

        handleCategoryClicked : function(event) {
            event.data.ol.children().toggle();
            event.data.downArrowSVG.toggle();
            event.data.rightArrowSVG.toggle();
        }
    };
})();

/**
 * the command group : openFile, openSheetSet etc.
 */
ntp.widget.commandsGroup = {
    options : {
        openFile : 'enabled',
        openSheetSet : 'enabled',
        exploreSamples : 'enabled',
        getOnlineTemplates : "",
        customLinks : []
    },

    _create : function() {
        var that = this;
        var $ul = $('<ul>').attr('id', "get_started_commands");
        // predefined commands
        if (this.options.openFile === 'enabled')
            ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_open', text : 'get_started_open_label', action : 'ntp.acad.showOpenFileDialog()', custom : false }).appendTo($ul);
        if (this.options.openSheetSet === 'enabled')
            ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_open_sheet_set', text : 'get_started_open_sheet_set_label', action : 'ntp.acad.showOpenSheetSetDialog()', custom : false }).appendTo($ul);
        if (this.options.getOnlineTemplates !== '' && this.options.getOnlineTemplates !== 'disabled') {
            var $getTemplates = ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_templates', text : 'get_started_templates_label', custom : false });
            ntp.deferred.localization.done(function() {
                $getTemplates.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'openUrlWithBrowser', ntp.utils.localize(that.options.getOnlineTemplates)));
            });
            $ul.append($getTemplates);
        }
        if (this.options.exploreSamples === 'enabled')
            ntp.widget.showWidgetWithOptions($('<div>'), 'Command', { id : 'get_started_samples', text : 'get_started_samples_label', action : 'ntp.acad.showOpenLocalSampleFileDialog()', custom : false }).appendTo($ul);
        // custom links
        this.processLinks(this.options.customLinks, $ul);
        this.element.append($ul);
    },

    processLinks : function(links, parent) {
        if (!( links instanceof Array)) {
            return;
        }

        $.each(links, function(i, link) {
            if (!( typeof link.text === 'string' && typeof link.action === 'string')) {
                return;
            }

            var $link = $('<li>').addClass('button');

            ntp.utils.localize($link, link.text);

            if ( typeof link.icon === 'string') {
                $link.css('background-image', 'url("' + ntp.utils.resolvePath(link.icon) + '")');
            }

            $link.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', link.action));
            parent.append($link);
        });
    }
};

/**
 * widget for command, e.g. openFile, openSheetSet, getSamples etc.
 */
ntp.widget.command = {
    options : {
        id : '', // required
        text : '', // required
        class_ : '', // optional
        icon : '', // optional
        action : '', // optional
        content : '', // optional
        custom : true
    },

    _create : function() {
        var $item = $('<li>').attr({
            'id' : this.options.id,
            'data-text-id' : this.options.text
        }).addClass('button').text('');
        if (this.options.action !== '' && !this.options.custom)
            $item.attr('onclick', this.options.action);
        else
            $item.click(ntp.utils.makeCallbackWithArgs(ntp.acad, 'invokeCustomAction', this.options.action));

        this.element.append($item);
    },

    _setOption : function(key, value) {
        switch( key ) {
            case "action":
                break;
        }
        $.Widget.prototype._setOption.apply(this, arguments);
        this._super("_setOption", key, value);
    },

    destroy : function() {
        $.Widget.prototype.destroy.call(this);
    }
};

/*
/* Define widget for Notification box that includes hardware acceleration, license information and live update.
*/
ntp.widget.notifications = (function() {
    var $notificationBox = null;
    var $updateBox = null;
    var $licenseBox = null;
    var $hwaccBox = null;
    var $containerBox = null;
    var $licenseImage = null;
    var $notificationSection = null;
    var $offlineHelpBox = null;
    var $notificationBadge = null;
    var $footerBadge = null;
    var notificationNum = 0;
    
    var getDayImage = function(day) {
        return 'images/day' + day.toString() + '.png';
    };

    return {
        //options
        options : {
            title : 'general_notifications',
            id : 'section_notifications',
            hwaccAction : 'javascript:ntp.acad.showGsConfigDialog();', //button click action.
            hwaccText : 'hardware_acceleration_message',
            hwaccBText : 'hardware_acceleration_title',
            updateText : 'general_update',
            trialText : 'trial_description_long',
            remainingText : 'trial_days_remaining',
            activeText : 'general_activate',
            activateAction : 'trial_days_remaining',
            updateAction: 'javascript:ntp.acad.showProductUpdateDialog();',
            Update : 'enabled',
            Hwacc : 'enabled',
            License : 'enabled',
            OfflineHelp : 'enabled',
            offlineHelpText : 'offline_help_message'
        },
  
        //the constructor
        _create : function() {
            this.element.attr({ 'id' : this.options.id }).addClass('hidden');
            $notificationSection = this.element;

            $('<h1>').attr({ 'id' : 'notification_title', 'data-text-id' : this.options.title }).appendTo(this.element);
            $notificationBadge = $('<div>').attr({'id' : 'notification_badge'}).html(notificationNum.toString()).addClass('badge hidden').appendTo(this.element);
            $footerBadge = $('#footer_create_badge');
            $containerBox = $('<div>').attr({ 'id' : 'notification_container'});
            $notificationBox = $('<div>').attr({ 'id' : 'notification_box'});
     
            this._initUpdateBox();
            this._initHwaccBox();
            this._initLicenseBox();
            this._initOfflineHelpBox();
            $notificationBox.appendTo($containerBox);
            $containerBox.appendTo(this.element);
        },
        
        _initHwaccBox : function() {
            if (this.options.Hwacc === 'enabled') {
                $hwaccBox = $('<div>').attr({ 'id' : "notification_hwacc" }).addClass('hidden shadow_box');

                $('<span>').attr({ 'data-text-id' : this.options.hwaccText }).addClass("description").appendTo($hwaccBox);
                $('<img>').attr({ 'alt' : "Hardware Acceleration", 'src' : "images/hardware_acc.png" }).appendTo($hwaccBox);
                $('<button>').attr({ 'id' : 'button_hardware_acceleration', 'data-text-id' : this.options.hwaccBText, 'onclick' : this.options.hwaccAction, 'tabindex' : '-1' }).addClass("blue").appendTo($hwaccBox);
                $hwaccBox.appendTo($notificationBox);

                //Set show/hide hardware acceleration notification based on acad information.
                this.beginGetHwaccStatus();
            }
        },
        
        _initUpdateBox : function() {
            if (this.options.Update === 'enabled') {
                $updateBox = $('<div>').attr({'id' : 'notification_update'}).addClass('hidden shadow_box');
                
                $('<span>').attr({'data-text-id' : 'update_description_critical'}).addClass("description").appendTo($updateBox);
                $('<img>').attr({ 'alt' : "Update", 'src' : "images/update.png" }).appendTo($updateBox);
                $('<button>').attr({'data-text-id' : this.options.updateText, 'id' : 'button_update', 'onclick' : this.options.updateAction, 'tabindex' : '-1'}).addClass("blue").appendTo($updateBox);
                
                $updateBox.appendTo($notificationBox);
                this.beginGetUpdateStatus();
            }
        },
        
        _initLicenseBox : function() {
            if (this.options.License === 'enabled') {
                $licenseBox = $('<div>').attr({'id' : 'notification_trial'}).addClass('hidden shadow_box');
                
                $('<span>').attr({'data-text-id' : this.options.trialText}).addClass("description").appendTo($licenseBox);
                $licenseImage = $('<img>').attr({ 'alt' : "Remaining Days", 'src' : "images/day30.png" });
                $licenseImage.appendTo($licenseBox);
                
                $licenseBox.appendTo($notificationBox);
                this.beginGetLicenseStatus();
            }
        },
        
        _initOfflineHelpBox : function() {
            if (this.options.OfflineHelp === 'enabled') {
                $offlineHelpBox = $('<div>').attr({'id' : 'offline_help'}).addClass('hidden shadow_box');
                
                $('<span>').attr({'data-text-id' : this.options.offlineHelpText}).addClass("description").appendTo($offlineHelpBox);
                $('<img>').attr({ 'alt' : "Offline Help", 'src' : "images/offline_help.png" }).appendTo($offlineHelpBox);
                
                $offlineHelpBox.appendTo($notificationBox);
            }
        },
        
        loadAlerts : function(url) {
            $.ajax({
                url : url,
                dataType : 'jsonp',
                jsonpCallback : 'alertCallback',
                crossDomain : true,
                timeout : 5000,
                success : function(data) {
                    ntp.acad.getViewedAlerts().done(function (obj) {
                        var viwedAlertData = $.parseJSON(obj);

                        if (typeof viwedAlertData.retValue !== 'undefined')
                            viwedAlertData = viwedAlertData.retValue; // hoist retValue if present

                        if (!(viwedAlertData.viewedAlerts instanceof Array))
                            viwedAlertData.viewedAlerts = []; // load all alerts

                        ntp.widget.notifications.getAlerts(data, viwedAlertData.viewedAlerts);
                    });
                },
                error : function() {
                }
            });
        },

        getAlerts: function (data, viewedAlerts) {

            if (data.alerts instanceof Array) {
                var alertsList = [];
                for (var i = 0; i < data.alerts.length; i++) {
                    var activeTime = new Date(data.alerts[i].active_time);
                    var id = Number(data.alerts[i].id);
                    var found = false;
                    for (var j = 0 ; j < viewedAlerts.length; j++) {
                        if (viewedAlerts[j].id === id) {
                            found = true; break;
                        }
                    }
                    if (found)
                        continue;
                    var expiryDays = data.alerts[i].expiry;
                    var today = new Date();
                    var expireDate = new Date(activeTime.getTime() + expiryDays * 24 * 60 * 60 * 1000);
                    if (activeTime < today && expireDate > today) {
                        var $alertBox = $('<div>').addClass('shadow_box');
                        var $description = $('<span>').addClass('description').text(data.alerts[i].description).appendTo($alertBox);
                        if (typeof data.alerts[i].image !== 'undefined' && data.alerts[i].image !== "") {
                            $('<img>').attr({ 'src': data.alerts[i].image }).appendTo($alertBox);
                        }
                        else {
                            $description.css('width', '100%');
                        }
                        $('<a>').text(data.alerts[i].link_text).on("click", { link: data.alerts[i].link, id: data.alerts[i].id, alertbox: $alertBox }, function (e) {

                            if (typeof e.data.link !== 'undefined' && e.data.link !== "" && e.data.id !== 'undefined' && e.data.id !== "") {
                                ntp.acad.openAlertLink(e.data.id, e.data.link);
                                // hide the clicked alert box
                                e.data.alertbox.addClass('hidden');
                            };
                        }).appendTo($alertBox);
                        // based on alert priority, it will added into different place.
                        if (data.alerts[i].priority == 1) {
                            $alertBox.addClass('high_priority');
                            if (i == 0) {
                                $alertBox.prependTo($notificationBox);
                            }
                            else {
                                $notificationBox.children().eq(i - 1).after($alertBox);
                            }
                        }
                        else if (data.alerts[i].priority == 3) {
                            $alertBox.addClass('low_priority');
                            $alertBox.appendTo($notificationBox);
                        }
                        else {
                            continue;
                        }
                        alertsList.push(data.alerts[i].id);
                    };
                };
                this.updateNotificationSection();
                if (alertsList.length > 0)
                    ntp.acad.getAlertViewState(alertsList).done(ntp.utils.makeCallback(this, 'refreshAlertsNumber'));
            };
        },

        refreshAlertsNumber : function(numberJson) {
            var data = $.parseJSON(numberJson);
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (data.number != 0) {
                this.updateBadgeNumber(data.number);
            }
        },

        hasNotifications : function() {
            var numbers = $notificationBox.children(':not(.hidden)').length;
            var ret;
            if (numbers === 0)
                ret = false;
            else
                ret = true;
            return ret;
        },
        
        adjustContainerHeight : function() {
            var $firstChild = $notificationBox.children(':not(.hidden)');
            if ($firstChild.length > 0) {
                $containerBox.css("max-height", $firstChild.eq(0).outerHeight(true));
            }
        },
        
        updateNotificationSection : function() {
            var number = $notificationBox.children(':not(.hidden)').length;
            if (number === 0) {
                $notificationSection.addClass('hidden');
            }
            else {
                $notificationSection.removeClass('hidden');
            }
            ntp.widget.notifications.adjustContainerHeight();
            ntp.page.layout.updateConnectHeight();
        },

        updateBadgeNumber : function(updateNumber) {
            notificationNum += updateNumber;
            if (notificationNum > 0) {
                $notificationBadge.html(notificationNum.toString());
                $footerBadge.html(notificationNum.toString());
                $notificationBadge.removeClass('hidden');
                $footerBadge.removeClass('hidden');
            } else{
                $notificationBadge.addClass('hidden');
                $footerBadge.addClass('hidden');
            };

        },
        
        beginGetUpdateStatus : function() {
            return ntp.acad.getUpdateInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshUpdateWithJSON'));
        },
        
        beginGetHwaccStatus : function() {
            return ntp.acad.getHardwareAccelerationInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshHwaccWithJSON'));
        },
        
        beginGetLicenseStatus : function() {
            return ntp.acad.getLicenseInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshLicenseWithJSON'));
        },
        
        beginGetOfflineHelpStatus : function() {
            return ntp.acad.getOfflineHelpInfoAsJSON().done(ntp.utils.makeCallback(this, 'refreshOfflineHelpWithJSON'));
        },
        
        refreshUpdateWithJSON : function(updateInfo) {
            var data = $.parseJSON(updateInfo); 
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.update === 0) {
                if (!$updateBox.hasClass('hidden')) {
                    $updateBox.addClass('hidden');
                    num = -1;
                }
            }
            else {
                if(data.update === 1)
                    var $updateTitle = $('span', $updateBox).text(ntp.utils.string.format(ntp.utils.localize('update_info_title'), data.update));
                else
                    var $updateTitle = $('span', $updateBox).text(ntp.utils.string.format(ntp.utils.localize('updates_info_title'), data.update));
                if ($updateBox.hasClass('hidden')) {
                    $updateBox.removeClass('hidden');
                    num = 1;
                }
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        },
        
        refreshHwaccWithJSON : function(hardwareAccelerationInfo) {
            var data = $.parseJSON(hardwareAccelerationInfo);
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.isHardwareAccelerationOn === true) {
                if (!$hwaccBox.hasClass('hidden')) {
                    $hwaccBox.addClass('hidden');
                    num = -1;
                }
            } else {
                if ($hwaccBox.hasClass('hidden')) {
                    $hwaccBox.removeClass('hidden');
                    num = 1;
                }
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        },
        
        refreshLicenseWithJSON : function(licenseInfo) {
            var data = $.parseJSON(licenseInfo);
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.type !== 2) {
                if (!$licenseBox.hasClass('hidden')) {
                    $licenseBox.addClass('hidden');
                    num = -1;
                }
            }
            else {
                if ($licenseBox.hasClass('hidden')) {
                    $licenseBox.removeClass('hidden');
                    num = 1;
                }
                if (data.daysleft >= 0)
                    $licenseImage.attr({'src' : getDayImage(data.daysleft)});
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        },
        
        refreshOfflineHelpWithJSON : function(offlineHelpInfo) {
            var data = $.parseJSON(offlineHelpInfo);
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            var num = 0;
            if (data.isOfflineHelpInstalled === true) {
                if (!$offlineHelpBox.hasClass('hidden')) {
                    $offlineHelpBox.addClass('hidden');
                    num = -1;
                }
            } else {
                if ($offlineHelpBox.hasClass('hidden')) {
                    $offlineHelpBox.removeClass('hidden');
                    num = 1;
                }
            }
            this.updateNotificationSection();
            if (num != 0)
                this.updateBadgeNumber(num);
        }
    };
})();

ntp.widget.connect = (function(){
    var $connectPanel = null; 
    var $feedbackPanel = null;
    var $userPanel = null;
    var $loginPanel = null;
    var $widgetPanel = null;
    var $scrollPanel = null;
    var embededWidgetURL = null;
    var embededWidgetEnable = null;
    var widgetIsOnline = false;
    var $image = null;
     
    return {
        options : {
            id : 'section_connect',
            title : 'general_connect',
            loginTitle : 'connect_login_title',
            loginSummary : 'connect_login_summary',
            loginAction : 'javascript:ntp.acad.showLoginDialog();',
            feedbackTitle : 'feedback_title',
            feedbackSummary : 'feedback_summary',
            feedbackAction : 'javascript:ntp.acad.sendFeedbackWithBrowser();',
            connectWelcome : 'connect_login_welcome',
            connectAction : 'javascript:ntp.acad.openUserHomePageWithBrowser()',
            feedbackPanel : 'enabled',
            userInfoPanel : 'enabled',
            contentWidgetPanel: 'enabled'
        },
        
        _create : function() {
            this.element.attr({ 'id' : this.options.id });
            if(!this.hasConnect())
                this.element.addClass('hidden');

            $('<h1>').attr({ 'data-text-id' : this.options.title, 'id' : 'connect_title'}).appendTo(this.element);

            $scrollPanel = $('<div>').attr({ 'id' : 'connect_scroll_panel'});
            $connectBox = $('<div>').attr({ 'id' : 'connect_box' }).addClass("login shadow_box white_box");
            if (this.options.userInfoPanel === 'enabled') {
                //connect panel.
                $connectPanel = $('<div>').attr({ 'id' : 'connect_user_panel' }).addClass('hidden');

                //login in panel.
                $loginPanel = $('<div>').attr({ 'id' : 'login_panel' }).addClass('hidden');

                $('<h2>').attr({ 'data-text-id' : this.options.loginTitle }).appendTo($loginPanel);
                $('<p>').attr({ 'data-text-id' : this.options.loginSummary }).appendTo($loginPanel);
                $('<button>').attr({ 'data-text-id' : 'connect_login_button', 'class' : 'blue login', 'onclick' : this.options.loginAction, 'tabindex' : '-1' }).appendTo($loginPanel);

                $loginPanel.appendTo($connectPanel);

                //user information panel.
                $userPanel = $('<div>').attr({ 'id' : 'user_info_panel' }).addClass('hidden');
            
                $('<h2>').attr({ 'id' : 'welcome_message', 'data-text-id' : this.options.connectWelcome }).appendTo($userPanel);
                $('<h2>').attr({ 'id' : 'user_name' }).appendTo($userPanel);
                $('<p>').attr({ 'id' : 'user_email' }).appendTo($userPanel);
                $('<p>').attr({ 'id' : 'closing_brace' }).html(')').appendTo($userPanel);
                $image = $('<img>').attr({ 'alt' : 'A360', }).appendTo($userPanel);

                $userPanel.appendTo($connectPanel);
                $connectPanel.appendTo($connectBox);
   
            }
            
            embededWidgetEnable = this.options.contentWidgetPanel;
            
            if (this.options.contentWidgetPanel === 'enabled') {
                $widgetPanel = $('<iframe>').attr({ 'id': 'content_widget_box' }).addClass("hidden");
                $widgetPanel.appendTo($connectBox);
            }

            if (this.options.feedbackPanel === 'enabled') {
                $feedbackPanel = $('<div>').attr({ 'id' : 'feedback_panel' });

                $('<h2>').attr({ 'data-text-id' : this.options.feedbackTitle }).appendTo($feedbackPanel);
                $('<p>').attr({ 'data-text-id' : this.options.feedbackSummary }).appendTo($feedbackPanel);
                $('<button>').attr({ 'data-text-id' : this.options.feedbackTitle, 'onclick' : this.options.feedbackAction, 'tabindex' : '-1' }).addClass('blue feedback').appendTo($feedbackPanel);

                $feedbackPanel.appendTo($connectBox);
            }
            $connectBox.appendTo($scrollPanel);
            $scrollPanel.appendTo(this.element);
        },
        
        hasConnect : function() {
            return this.options.feedbackPanel === 'enabled' || this.options.userInfoPanel === 'enabled' || this.options.contentWidgetPanel === 'enabled';
        },
        
        isEnabled : function() {
            return this.options.userInfoPanel !== 'disabled';
        },
        
        setWidgetURL : function(url) {
            embededWidgetURL = url;
        },
        
        setIsOnline : function(isOnline) {
            widgetIsOnline = isOnline;
            if (isOnline) {
                if($image !== null) {
                    $image.attr({'src' : 'images/A360_logo.png', 'onclick' : this.options.connectAction } );
                    $image.css("cursor", "URL");
                }
            }
            else {
                if ($image !== null) {
                    $image.attr('src', 'images/A360_logo_offline.png');
                    $image.css("cursor", "default");
                }
            }
        },

        showWidgetPanel : function() {
            if ($connectPanel !== null)
                $connectPanel.addClass('hidden');
            if ($widgetPanel !== null)
                $widgetPanel.removeClass('hidden');
        },
        
        showLoginPanel : function() {
            if ($userPanel !== null)
                $userPanel.addClass('hidden');
            if ($widgetPanel !== null)
                $widgetPanel.addClass('hidden');
            if ($connectPanel !== null)
                $connectPanel.removeClass('hidden');
            if ($loginPanel !== null)
                $loginPanel.removeClass('hidden');
        },
        
        showUserInfoPanel : function(data) {
            if ($connectPanel === null)
                return;
            $("#closing_brace").addClass('hidden');
            $loginPanel.addClass('hidden');
            $connectPanel.removeClass('hidden');
            $userPanel.removeClass('hidden');
            if ($widgetPanel !== null)
                $widgetPanel.addClass('hidden');
            var $welcomeMsg = $('#welcome_message');
            var $userName = $("#user_name");
            var $userEmail = $("#user_email");

            if (data.loggedInCurrentSession) {
                var userName = data.userName;
                var userEmail = data.userEmail;
                var userFirstName = data.userFirstName;

                // Show the welcome message part
                $welcomeMsg.removeClass('hidden');
                $userEmail.removeClass('hidden');
                $userName.removeClass('hidden');

                $userName.text(" " + userFirstName + "!");
                if ($userName[0].scrollWidth > $userName.width()) {
                    $userName.attr('data-title', $userName.text());
                } else {
                    $userName.attr('data-title', '');
                }
                $userName.hover(showTooltip, hideTooltip);

                $userEmail.text("(" + userEmail + ")");
                if ($userEmail[0].scrollWidth > $userEmail.width()) {
                    $userEmail.attr('data-title', $userEmail.text());
                    $("#closing_brace").removeClass('hidden');
                } else {
                    $userEmail.attr('data-title', '');
                }
                $userEmail.hover(showTooltip, hideTooltip);
            } else {
                // Hide the welcome message part
                $welcomeMsg.addClass('hidden');
                $userEmail.addClass('hidden');
                $userName.addClass('hidden');
            }
        },
        
        refresh : function(data) {
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (data.isLoggedIn === 'undefined')
                return;
            if ( data.isLoggedIn === true) {
                $connectPanel.addClass('hidden');
                if (embededWidgetEnable === 'enabled' && embededWidgetURL !== null){
                    $widgetPanel.attr("src", data.signedUrl);
                    ntp.widget.connect.showWidgetPanel();
                }
                else{
                    ntp.widget.connect.showUserInfoPanel(data);
                }
            }
            else {
                ntp.widget.connect.showLoginPanel();
            }
            ntp.page.layout.updateConnectHeight();
        },
        
        beginRefresh : function() {
            if (!this.isEnabled()) {
                return null;
            }
            return ntp.acad.getSignedRequestDataAsJSON(embededWidgetURL).done(ntp.utils.makeCallback(this, 'refreshWithJSON'));
        },
        
        refreshWithJSON : function(userInfoJSON) {
            if (!this.isEnabled()) {
                return;
            }
            this.refresh($.parseJSON(userInfoJSON));
        },
        
        refreshLoginWithJSON : function(data) {
            ntp.asyncLoadings.push(this.beginRefresh());
        }
    };
})();

ntp.widget.continueWorking = (function() {
    var $recentFiles = null;
    
    var setViewName = function(viewName) {
        $('#recent_file_view_options li').each(function() {
            var $viewOptionButton = $(this);
            var viewOptionName = $viewOptionButton.data('view-name');
            if (viewOptionName !== viewName) {
                $viewOptionButton.removeClass('selected');
                $recentFiles.removeClass(viewOptionName);
            } else {
                $viewOptionButton.addClass('selected');
            }
        });
        $recentFiles.addClass(viewName);
    };
    
    var handleRecentFileClicked = function() {
        ntp.acad.openDocumentWithPath($(this).data('full-path'));
    };
    
    var recentFileCategories = {
        0 : 'pinned',
        1 : 'timing_today',
        2 : 'timing_yesterday',
        3 : 'timing_this_week',
        4 : 'timing_last_week',
        5 : 'timing_this_month',
        6 : 'timing_last_month',
        7 : 'timing_older'
    };
    
    var resetToMidnight = function(d) {
        d.setHours(0, 0, 0, 0);
    };

    var listCategories = function() {
        return recentFileCategories;
    };

    //Implements pin animation of recent files.
    var pinningInProgress = false;
    
    var createCategory = function(categoryId) {
        var categoryName = recentFileCategories[categoryId];
        var categoryText = ntp.utils.localize(categoryName);
        if (categoryText === undefined) {
            // Ensure <h2> element is created to reduce number of edge cases in pin/unpin animation
            categoryText = '';
        }
        return $('<li>').data('category-id', categoryId).append($('<h2>').text(categoryText)).append($('<ol>'));
    };
    
    var handlePinButtonClicked = function(event) {
        event.stopPropagation();

        if (pinningInProgress) {
            return;
        }
        pinningInProgress = true;

        var $currentDoc = $(this).parent().parent();
        var fullPath = $currentDoc.data('full-path');
        var $currentCategory = $currentDoc.parent().parent();
        var $currentCategoryHeader = $currentCategory.children('h2');

        // Update CSS class
        var isPinning = !($(this).hasClass('pinned'));
        if (isPinning) {
            $(this).addClass('pinned');
            $(this).parent().parent().addClass('pinned');
        } else {
            $(this).removeClass('pinned');
            $(this).parent().parent().removeClass('pinned');
        }

        //
        // Show pin/unpin animation
        //

        // Step 1, determine the elements that needs to be moved.
        var $targetCategory = null;

        var shouldRemoveCurrentCategory = $currentDoc.siblings().length === 0;

        var $currentCategoryDocsInBetween = isPinning ? $currentDoc.prevAll() : $currentDoc.nextAll();
        var $currentCategoryDocsBelow = isPinning ? $currentDoc.nextAll() : $([]);
        var targetCategoryId = ntp.widget.continueWorking.getCategory($currentDoc.data('last-opened'), isPinning);
        var currentCategoryId = $currentCategory.data('category-id');

        var $categoriesInBetween = $([]);
        var $categoriesBelow = $([]);

        var $prevCategory = null;
        $recentFiles.children().each(function() {
            var $thisCategory = $(this);
            var categoryId = $thisCategory.data('category-id');
            if (categoryId === targetCategoryId) {
                $targetCategory = $thisCategory;
            } else if (categoryId < targetCategoryId) {
                $prevCategory = $thisCategory;
                if (categoryId > 0) {
                    $categoriesInBetween = $categoriesInBetween.add($thisCategory);
                }
            } else {// categoryId > targetCategoryId
                if (isPinning && categoryId > 0 && categoryId < currentCategoryId) {
                    $categoriesInBetween = $categoriesInBetween.add($thisCategory);
                } else if (categoryId > currentCategoryId) {
                    $categoriesBelow = $categoriesBelow.add($thisCategory);
                }
            }

            if (isPinning && categoryId > 0 && categoryId < currentCategoryId) {
                $categoriesInBetween = $categoriesInBetween.add($thisCategory);
            }
        });

        var $targetCategoryDocsInBetween = $([]);
        var $targetCategoryDocsBelow = $([]);
        var shouldCreateTargetCategory = ($targetCategory === null);
        var $targetCategoryHeader = null;
        if ($targetCategory !== null) {
            $targetCategoryHeader = $targetCategory.children('h2');
        }

        var $prevDoc = null;
        var $targetList = null;
        if (!shouldCreateTargetCategory) {
            $targetList = $targetCategory.children('ol');

            if ($targetList.children().length > 0) {
                var lastOpened = $currentDoc.data('last-opened');
                $targetList.children().each(function() {
                    var $thisDoc = $(this);
                    if ($thisDoc.data('last-opened') > lastOpened) {
                        $prevDoc = $thisDoc;
                        if (!isPinning) {
                            $targetCategoryDocsInBetween = $targetCategoryDocsInBetween.add($thisDoc);
                        }
                    } else {
                        if (isPinning) {
                            $targetCategoryDocsInBetween = $targetCategoryDocsInBetween.add($thisDoc);
                        } else {
                            $targetCategoryDocsBelow = $targetCategoryDocsBelow.add($thisDoc);
                        }
                    }
                });
            }
        }

        // Step 2, calculate element movement
        var $elementsInBetween = $currentCategoryDocsInBetween.add($categoriesInBetween).add($targetCategoryDocsInBetween);

        if (isPinning && $currentCategoryHeader.length > 0 && $currentCategoryHeader.is(':visible')) {
            $elementsInBetween = $elementsInBetween.add($currentCategoryHeader);
        } else if (!isPinning && !shouldCreateTargetCategory) {
            $elementsInBetween = $elementsInBetween.add($targetCategoryHeader);
        }

        var elementsInBetweenTopChange = isPinning ? $currentDoc.outerHeight(true) : -$currentDoc.outerHeight(true);

        var currentDocTopChange = 0;
        $elementsInBetween.each(function() {
            currentDocTopChange += $(this).outerHeight(true);
        });
        if (isPinning) {
            currentDocTopChange = -currentDocTopChange;
        }

        var $elementsBelow = $categoriesBelow.add($targetCategoryDocsBelow).add($currentCategoryDocsBelow);
        var elementsBelowTopChange = 0;

        var headerHeight = $recentFiles.find('li:last-child > h2').outerHeight(true);

        if (isPinning) {
            if (shouldRemoveCurrentCategory) {
                elementsBelowTopChange += -headerHeight;
            }
            if (shouldCreateTargetCategory) {
                elementsInBetweenTopChange += headerHeight;
                elementsBelowTopChange += headerHeight;
            }
        } else {
            if (shouldRemoveCurrentCategory) {
                elementsBelowTopChange += -headerHeight;
                elementsInBetweenTopChange += -headerHeight;
                currentDocTopChange += -headerHeight;
            }
            if (shouldCreateTargetCategory) {
                elementsBelowTopChange += headerHeight;
                currentDocTopChange += headerHeight;
            }
        }

        // Step 3, animate elements
        var animationCount = 0;
        var updateDomTree = function() {
            animationCount--;
            if (animationCount > 0) {
                return;
            }

            if (shouldCreateTargetCategory) {
                $targetCategory = createCategory(targetCategoryId);
                $targetCategoryHeader = $targetCategory.children('h2');
                if ($prevCategory !== null) {
                    $prevCategory.after($targetCategory);
                } else {
                    $recentFiles.prepend($targetCategory);
                }

                $targetList = $targetCategory.children('ol');
            }

            if ($prevDoc !== null) {
                $prevDoc.after($currentDoc);
            } else {
                $targetList.prepend($currentDoc);
            }

            if (shouldRemoveCurrentCategory) {
                $currentCategory.remove();
            }

            var $secondHeader = $recentFiles.find('li:nth-child(2) > h2');
            if ($secondHeader.length > 0) {
                if (shouldCreateTargetCategory && isPinning && elementsBelowTopChange !== 0) {
                    $secondHeader.css('opacity', 0);
                    $secondHeader.animate({
                        'opacity' : 1
                    });
                } else {
                    // Force refresh the second header
                    $secondHeader.css('display', 'none');
                    $secondHeader.css('display', '');
                }
            }
            if (shouldCreateTargetCategory && !isPinning) {
                // Simple fadeIn have bugs
                $targetCategoryHeader.css('opacity', 0);
                $targetCategoryHeader.animate({
                    'opacity' : 1
                });
            }

            pinningInProgress = false;

            // Only updates pinned data in AutoCAD after the animation is done.
            // Updating AutoCAD will result in refreshing all the currently opened NTPs
            // which slows down the animation and makes it laggy.
            //
            // Update pinned data in AutoCAD. Made it async so that problems with this
            // call does not impact the pin animation
            setTimeout(function() {
                ntp.acad.pinDocumentWithPath(fullPath, isPinning);
            }, 0);

        };

        if (shouldRemoveCurrentCategory) {
            var $secondHeader = $recentFiles.find('li:nth-child(2) > h2');
            if (isPinning && $currentCategoryHeader.is(':visible')) {
                $elementsInBetween = $elementsInBetween.not($currentCategoryHeader);
                $currentCategoryHeader.css('opacity', 1);
                $currentCategoryHeader.animate({
                    'opacity' : 0
                });
            }

            if (!isPinning && $secondHeader.length > 0) {
                $elementsInBetween = $elementsInBetween.not($secondHeader);

                if (elementsBelowTopChange !== 0) {
                    $secondHeader.css('opacity', 1);
                    $secondHeader.animate({
                        'opacity' : 0
                    });
                }
            }
        }

        var $lastAnimation = null;
        if ($elementsInBetween.length > 0 && elementsInBetweenTopChange !== 0) {
            animationCount += $elementsInBetween.length;
            $elementsInBetween.css('position', 'relative');
            $lastAnimation = $elementsInBetween.animate({
                top : elementsInBetweenTopChange
            }, function() {
                $(this).css({
                    top : '',
                    position : ''
                });

                updateDomTree();
            });
        }

        if ($elementsBelow.length > 0 && elementsBelowTopChange !== 0) {
            animationCount += $elementsBelow.length;
            $elementsBelow.css('position', 'relative');
            $lastAnimation = $elementsBelow.animate({
                top : elementsBelowTopChange
            }, function() {
                $(this).css({
                    top : '',
                    position : ''
                });

                updateDomTree();
            });
        }

        if (currentDocTopChange !== 0) {
            animationCount += $currentDoc.length;
            $currentDoc.css('position', 'relative');
            $currentDoc.css('top', 0);
            $currentDoc.animate({
                top : currentDocTopChange
            }, function() {
                $(this).css({
                    top : '',
                    position : ''
                });

                updateDomTree();
            });
        }

        if (animationCount === 0) {
            updateDomTree();
        }
    };
    
    var categorizeRecentFiles = function(data) {
        var recentFiles = data;
        var categorizedRecentFiles = [];

        var i;
        var categories = listCategories();
        var ids = Object.keys(categories);
        for ( i = 0; i < ids.length; i++) {
            categorizedRecentFiles[ids[i]] = [];
        }

        for ( i = 0; i < recentFiles.length; i++) {
            var lastOpened = new Date(recentFiles[i].lastOpened);
            recentFiles[i].lastOpened = lastOpened;
            recentFiles[i].lastOpenedText = recentFiles[i].lastOpenedText;
        }

        recentFiles.sort(function(a, b) {
            return b.lastOpened.getTime() - a.lastOpened.getTime();
        });

        for ( i = 0; i < recentFiles.length; i++) {
            var id = ntp.widget.continueWorking.getCategory(recentFiles[i].lastOpened, recentFiles[i].pinned);
            categorizedRecentFiles[id].push(recentFiles[i]);
        }

        return categorizedRecentFiles;
    };
    
    var loadCategorizedRecentFiles = function(categorizedRecentFiles) {
        // Add recent files into page
        $.each(categorizedRecentFiles, function(categoryId, recentFiles) {
            if (recentFiles.length == 0) {
                return;
            }

            var $category = createCategory(categoryId);
            var $ol = $category.children('ol');

            $.each(recentFiles, function(i, recentFile) {
                if ( typeof recentFile.type === 'undefined') {
                    recentFile.type = 'dwg';
                }

                var $thumb = $('<img>').addClass('thumb').attr('alt', 'thumbnail');
                if ( typeof recentFile.preview === 'string' && recentFile.preview !== '') {
                    $thumb.attr('src', 'data:image/png;base64,' + recentFile.preview);
                } else {
                    $thumb.attr('src', ntp.content.thumbnail(recentFile.type));
                }

                var $pin = $('<div class="pin">');
                if (recentFile.pinned === true) {
                    $pin.addClass('pinned');
                }
                //Event handler for pin/unpin a document
                $pin.click(handlePinButtonClicked);

                var $icon = $('<div>').addClass(recentFile.type).addClass('icon');
                var $title = $('<div>').addClass('title').text(recentFile.name);
                var lastOpenedBy = ntp.utils.localize('last_opened') + ' ' + recentFile.lastOpenedText;
                var $date = $('<div>').addClass('date').text(lastOpenedBy);
                var $imageContainer = $('<div>').addClass('image_container').append($thumb, $icon, $pin);
                
                if (recentFile.local !== true) {
                    $icon.addClass('cloudIcon');
                    $imageContainer.append($('<div>').addClass('cloud'));
                }
                // version is not added yet
                $li = $('<li>').data('full-path', recentFile.fullPath).data('last-opened', recentFile.lastOpened).append($imageContainer, $('<div>').addClass('details').append($title, $date), $('<div>').css('clear', 'both'));
                $li.attr('data-title', recentFile.fullPath);
                $li.hover(showTooltip, hideTooltip);

                if (recentFile.pinned === true) {
                    $li.addClass('pinned');
                }

                //Event handler for open a recent file
                $li.click(handleRecentFileClicked);

                $ol.append($li);
                // To make the <li> element fully contain the floating image
            });

            $recentFiles.append($category);
        });
    };
    
    var startLoadingRecentFile = function() {
        $recentFiles.addClass('disabled loading');
    };

    var endLoadingRecentFile = function() {
        $recentFiles.removeClass('disabled loading');
    };

    var processRecentFileImages = function(recentFileImages) {
        if (!( recentFileImages instanceof Array)) {
            return;
        }

        $.each(recentFileImages, function(i, recentFileImage) {
            if (!( typeof recentFileImage.filetype === 'string' && typeof recentFileImage.icon === 'string')) {
                return;
            }

            // Add rule for file type icon
            ntp.utils.addRule('#recent_files .' + recentFileImage.filetype + '{' + ' background-image: url("' + ntp.utils.resolvePath(recentFileImage.icon) + '");' + '}');

            // Store preview image
            if ( typeof recentFileImage.thumbnail === 'string') {
                ntp.content.thumbnail(recentFileImage.filetype, recentFileImage.thumbnail);
            }
        });
    };
    
    return {
        options : {
            id : 'continue_working',
            viewOptionId : 'recent_file_view_options',
            titleId : 'column_continue_working_title',
            minHeight : '504px',
           recentFileImages : []
        },
    
        _create : function() {
            this.element.attr({'id' : this.options.id});

            $('<h1>').attr({ 'data-text-id' : this.options.titleId }).appendTo(this.element);
            $recentFiles = $('<ol>').attr({ 'id' : 'recent_files' }).addClass("flex image_text");
            $recentFiles.appendTo(this.element);
            $viewOptions = $('<ul>').attr({ 'id' : this.options.viewOptionId }).addClass("view_options");

            var view = ['image', 'image_text', 'text'];
            for (var i = 0; i < view.length; i++) {
                var classtype = "button";
                $('<li>').attr({
                    'id' : 'recent_file_' + view[i] + '_view',
                    'data-view-name' : view[i]
                }).addClass(classtype).bind('click', function() {
                    ntp.widget.continueWorking.selectView($(this).data('view-name'));
                }).appendTo($viewOptions);
            }

            $viewOptions.appendTo(this.element);
            
            if (this.options.recentFileImages) {
                processRecentFileImages(this.options.recentFileImages);
            }
            
            this.beginRefresh();
        },
        
        getCategory : function(date, pinned) {
            if (pinned === true) {
                return 0;
            }
            var today = new Date();
            //today
            resetToMidnight(today);
            if (date >= today) {
                return 1;
            }

            var yesterday = new Date();
            //yesterday
            resetToMidnight(yesterday);
            yesterday.setDate(today.getDate() - 1);
            if (date >= yesterday) {
                return 2;
            }

            var thisWeek = new Date();
            //this week
            resetToMidnight(thisWeek);
            thisWeek.setDate(today.getDate() - today.getDay());
            if (date >= thisWeek) {
                return 3;
            }

            var lastWeek = new Date();
            //last week
            resetToMidnight(lastWeek);
            lastWeek.setDate(thisWeek.getDate() - 7);
            if (date >= lastWeek) {
                return 4;
            }

            var thisMonth = new Date();
            // this month
            resetToMidnight(thisMonth);
            thisMonth.setDate(1);
            // first day of this month
            if (date >= thisMonth) {
                return 5;
            }

            var lastMonth = new Date();
            //last month
            resetToMidnight(lastMonth);
            lastMonth.setMonth(thisMonth.getMonth() - 1, 1);
            //first day of last month
            if (date >= lastMonth) {
                return 6;
            }
            return 7;
        },

        selectView : function(viewName) {
            // Fade out recent files list
            // DID 148501, due to some display overlapping, we moved animation away from callback function.
            $recentFiles.stop().animate(
                {opacity : 0}, 
                ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION /2 , 
                'linear', function () {
                    // Close all opened version menus.
                    $('#recent_files .versions_menu:visible').each(function () {
                        $(this).hide();
                    });
            });

            // Change the class of recent files and track the change of height to make a smooth transition.
            $recentFiles.height('auto');
            // Clear previous fixed height setting
            var oldHeight = $recentFiles.height();
            setViewName(viewName);
            // Update recent file class
            var newHeight = $recentFiles.height();
            $recentFiles.height(oldHeight);

            // Fade in the recent file list with smooth transition of height.
            $recentFiles.stop().transition({
                opacity: 1,
                height: newHeight
            }, ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);

            ntp.acad.updateUserSetting("NewTabPageViewStyle", viewName);
        },

        refresh : function(data) {
            if ( typeof data === 'undefined' )
                return;
            if ( typeof data.retValue !== 'undefined' )
                data = data.retValue; // hoist retValue if present
            if ( typeof data.docs === 'undefined' )
                return;
            // Cleans old recent files
            $recentFiles.empty();

            // Categorizes recent file data by date
            var categorizedRecentFiles = categorizeRecentFiles(data.docs);

            // Load categorized data
            loadCategorizedRecentFiles(categorizedRecentFiles);
        },

        beginRefresh : function() {
            startLoadingRecentFile();

            ntp.acad.getUserSettingAsJSON("NewTabPageViewStyle").done(ntp.utils.makeCallback(this, 'refreshViewStyleWithJSON'));
            
            // Update recent file list
            var promise = ntp.acad.loadRecentFilesAsJSON().done(ntp.utils.makeCallback(this, 'refreshWithJSON')).always(endLoadingRecentFile);

            // If aync load of recent file list timed out, close
            // the busy indication.
            ntp.utils.timeout(endLoadingRecentFile, 10000);

            return promise;
        },

        refreshWithJSON : function(recentFilesJSON) {
            try {
                startLoadingRecentFile();
                this.refresh($.parseJSON(recentFilesJSON));
            } finally {
                endLoadingRecentFile();
            }
        },
        
        refreshViewStyleWithJSON : function(viewStyleJSON) {
            var data = $.parseJSON(viewStyleJSON);
            if (typeof data === 'undefined')
                return;
            if (typeof data.retValue !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if ( typeof data.NewTabPageViewStyle === 'undefined')
                return;
            var viewName = data.NewTabPageViewStyle;
            setViewName(viewName);
        }
    };
})();

ntp.widget.projectbar = (function() {
    var $projectBar = null;
    var $projectPanel = null;
    var $projectList = null;
    var $projectFooter = null;
    var $currentProject = null;
    var $openProject = null;
    var $newProject = null;
    var toggling = false;
    
    var handleClickedOutsideProjectPanel = function() {
        // Check if the mouse if hovering anywhere outside the panel
        if (!$projectList.is(':hover') && !toggling) {
            $projectPanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).off('click', handleClickedOutsideProjectPanel);
        }
        toggling = false;
    };
    
    var handleProjectBarClicked = function(event) {
        if ($projectPanel.css('display') === 'none') {
            toggling = true;
            updateFloatingProjectPanelHeight();
            
            $projectPanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
            $(document).on('click', handleClickedOutsideProjectPanel);
        }
    };
    
    var handleProjectClicked = function(event) {
        $projectPanel.slideToggle(ntp.page.animationSettings.ELEMENT_TOGGLING_ANIMATION_DURATION);
        $(document).off('click', handleClickedOutsideProjectPanel);
        
        ntp.acad.handleProjectClicked($(this).attr('data-title'));
        $(this).attr('title', '');
    };
    
    var createProjectItem = function (project) {
        if (project !== null) {
            var $li = $('<li>').addClass('button').attr('data-title', project.fullPath).text(project.name).click(handleProjectClicked);
            $li.hover(showTooltip, hideTooltip);
            return $li;
        }
        return null;
    };
    
    var addSepratorIfNeeded = function (projects) {
        if (projects.length >= 1 && $projectFooter !== null) {
             $projectFooter.before($('<hr>').css('margin-top', '3px'));
        }
    };
    
    var setCurrentProject = function (projectName) {
        if (projectName !== '') 
            $currentProject.text(projectName);
        else
            $currentProject.text(ntp.utils.localize('no_active_project'));
    };
    
    var loadProjects = function (projects) {
        $projectList.empty();
        $('hr', $projectPanel).remove();
        $.each(projects, function(i, project) {
            var $li = createProjectItem(project);
            $li.appendTo($projectList);
        });        
        addSepratorIfNeeded(projects);
    };
    
    var updateFloatingProjectPanelHeight = function() {
        // Calculate the maximum height of the panel
        var $tabCreate = $('#tab_create');
        var projectPanelHeight = $tabCreate.offset().top + $tabCreate.height() - $projectBar.offset().top - $projectBar.outerHeight() - 31;
        
        var projectFooterHeight = 0;
        if ($projectFooter !== null) {
            projectFooterHeight = $projectFooter.outerHeight(true);
            if ($projectPanel.css('display') === 'none') {
                $projectPanel.css({ 'display' : 'block', 'height' : 0 });
                projectFooterHeight = $projectFooter.outerHeight(true);
                $projectPanel.css({ 'display' : 'none', 'height' : 'auto'});
            }
        }
        $projectList.css('max-height', Math.min(projectPanelHeight - projectFooterHeight, 200));
    };
    
    return {
        options : {
            "projectBar" : "disabled",
            "newProject" : "enabled",
            "openProject" : "enabled",
            "projectIcon" : ""
        },
        
        _create : function() {
            if (this.options.projectBar === 'enabled') {
                $projectBar = $('<div>').attr('id', 'project_dropdown').addClass('button shadow_box').click(handleProjectBarClicked);
                if (this.options.projectIcon !== '') {
                    $projectBar.css('background-image', 'url("' + ntp.utils.resolvePath(this.options.projectIcon) + '")' + ',-webkit-linear-gradient(#e5e5e5,#c8c8c8)');
                    $projectBar.css('background-position','7px center, top left');
                }
                
                $currentProject = $('<span>').attr({'id': 'current_project', 'data-text-id': 'no_active_project'}).addClass('label').appendTo($projectBar);
                $('<span>').addClass('arrow').html('&#x25BC;').appendTo($projectBar);
                
                $projectPanel = $('<div>').attr('id', 'project_panel').addClass('shadow_box');
                $projectList = $('<ol>').attr('id', 'project_list').appendTo($projectPanel);
                var $ol = $('<ol>');
                if (this.options.openProject === 'enabled')
                    $openProject = $('<li>').attr('data-text-id', 'open_project').addClass('button').click(ntp.acad.showProjectNavigator).appendTo($ol);
                    
                if (this.options.newProject === 'enabled')
                    $newProject = $('<li>').attr('data-text-id', 'create_new_project').addClass('button').click(ntp.acad.newProject).appendTo($ol);;
                
                if ($openProject !==  null || $newProject !== null)
                    $projectFooter = $('<footer>').attr('id', 'project_footer').append($ol).appendTo($projectPanel);
                $projectPanel.appendTo($projectBar);
                
                this.element.append($projectBar);
                var projectBarRefresh = this.beginRefresh();
                ntp.asyncLoadings.push(projectBarRefresh);
            }
        },
        
        isEnabled : function () {
            return $projectBar !== null;
        },
        
        update: function() {
            if (this.isEnabled())
                updateFloatingProjectPanelHeight();
        },
        
        beginRefresh : function() {
            if (!this.isEnabled()) {
                return null;
            }
            ntp.api.loadProjectsAsJSON().done(ntp.utils.makeCallback(this, 'refreshWithJSON'));
            return ntp.acad.loadCurrentProjectAsJSON().done(ntp.utils.makeCallback(this, 'refreshCurrentProjectWithJSON'));
        },
        
        refreshWithJSON : function(projectsJSON) {
            this.refresh($.parseJSON(projectsJSON));
        },
        
        refresh: function (data) {
            if (!this.isEnabled())
                return;
            if (typeof data === 'undefined')
                return;
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (typeof (data.projects) === "undefined")
                return;
            loadProjects(data.projects);
        },
        
        setCurrentProject : function(data)
        {
            if (!this.isEnabled())
                return;
            if (typeof data === 'undefined')
                return;
            if (typeof (data.retValue) !== 'undefined')
                data = data.retValue; // hoist retValue if present
            if (typeof (data.name) === "undefined")
                return;
            
            setCurrentProject(data.name);
        },
        
        refreshCurrentProjectWithJSON : function(currentProjectJSON) {
            this.setCurrentProject($.parseJSON(currentProjectJSON));
        }
    };
})();

/**
 * setup all the widgets
 */
( function($) {
        $.widget("ntp.Title", ntp.widget.title);
        $.widget("ntp.Command", ntp.widget.command);
        $.widget("ntp.Templatebar", ntp.widget.templatebar);
        $.widget("ntp.GetStartedBox", ntp.widget.tile);
        $.widget("ntp.GetStarted", ntp.widget.getStarted);
        $.widget("ntp.CommandsGroup", ntp.widget.commandsGroup);
        $.widget("ntp.External", ntp.widget.external);
        $.widget("ntp.Notifications", ntp.widget.notifications);
        $.widget("ntp.Connect", ntp.widget.connect);
        $.widget("ntp.ContinueWorking", ntp.widget.continueWorking);
        $.widget("ntp.ProjectBar", ntp.widget.projectbar);
    }(jQuery));

/**
 * extend ntp.widget for loading widgets
 */
(function() {
    $.extend(ntp.widget, {
        learnPage : [],
        createPage : ["#column_get_started", "#column_continue_working", "#column_online"],

        loadWidgets : function(widgets) {
            var that = this;
            if ( typeof widgets.create === 'object' && widgets.create instanceof Array) {
                $.each(widgets.create, function(index, column) {
                    if ( typeof column === 'object' && column instanceof Array) {
                        $.each(column, function(i, w) {
                            if ( typeof w === 'object') {
                                $.each(w, function(widget, options) {
                                    if ( typeof options === 'string' && options === '') {
                                        ntp.widget.showWidget($('<div>'), widget).appendTo($(that.createPage[index]));
                                    } else {
                                        ntp.widget.showWidgetWithOptions($('<div>'), widget, options).appendTo($(that.createPage[index]));
                                    }
                                });
                            }
                        });
                    }
                });
            }
        },

        merge : function(common, spec) {
            // 1. adjust the order if it is specified in spec
            if ( typeof spec.placement !== 'undefined') {
                $.each(spec.placement.create, function(i, val) {
                    var tmp = [];
                    if ( val instanceof Array) {
                        $.each(val, function(j, v) {
                            if ( typeof v === 'string') {
                                var w = {};
                                w[v] = find(common.widgets.create, v);
                                tmp.push(w);
                            } else
                                tmp.push(v);
                        });
                    }
                    common.widgets.create[i] = tmp;
                });
            }
            // 2. update the property value in common with spec.
            $.each(spec, function(k, v) {
                update(common, k, v);
            });

            function find(obj, key) {
                var res = {};
                (function find_(obj, key) {
                    $.each(obj, function(k, v) {
                        if ( typeof k === 'string' && k === key) {
                            res = v;
                        }
                        if ( typeof v === 'object')
                            find_(v, key);
                    });
                })(obj, key);

                return res;
            }

            function update(obj, key, value) {
                $.each(obj, function(k, v) {
                    if ( typeof k === 'string' && k === key) {
                        obj[k] = value;
                        return false;
                    }
                    if ( typeof v === 'object')
                        update(v, key, value);
                });
            }

        },

        showWidget : function(container, widget) {
            return this.showWidgetWithOptions(container, widget, {});
        },

        showWidgetWithOptions : function(container, widget, options) {
            if ( typeof container == "string") {
                ntp.utils.makeCallbackWithArgs($(container), widget, options)();
                return $(container);
            } else if ( typeof container == "object") {
                ntp.utils.makeCallbackWithArgs(container, widget, options)();
                return container;
            }
        },

        applySettings : function(generic, spec) {
            if ( typeof generic.widgets !== 'undefined') {
                if (spec !== null) {
                    ntp.widget.merge(generic, spec);
                }
                this.loadWidgets(generic.widgets);
                if (spec !== null) {
                    if (typeof spec.alert !== 'undefined' && typeof spec.alert.enabled === 'string' && spec.alert.enabled.toLowerCase() === 'yes') {
                        //Load alerts from server JSON file.
                        if (typeof spec.alert.alertUrl === 'string') {
                            ntp.deferred.localization.done(function() {
                                ntp.widget.notifications.loadAlerts(ntp.utils.localize(spec.alert.alertUrl));
                            });
                        };
                    }
                }
            }
        },

        bindToolTips: function () {
            $("[data-title]").hover(showTooltip, hideTooltip);
        },

        unbindToolTips: function () {
            //Done on document switch out of new tabpage
            $("[data-title]").mouseleave().unbind('mouseenter mouseleave');
        }
    });
})();

// SIG // Begin signature block
// SIG // MIIZ4gYJKoZIhvcNAQcCoIIZ0zCCGc8CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // IZT9pINR1faV+wl69HqCXmYOd8W9Y94vzqYrLqRyVr+g
// SIG // ggpSMIIFAzCCA+ugAwIBAgIQdnK0JhTS6n5kEFOlgziE
// SIG // ljANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNjA4MDgwMDAw
// SIG // MDBaFw0xNzA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIElu
// SIG // YzEfMB0GA1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91
// SIG // cDEWMBQGA1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARq
// SIG // d9dP4fq2qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8
// SIG // WR/0tW17lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpR
// SIG // IG9QthGUq8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGV
// SIG // plDOyr976ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm
// SIG // +eRliYKeKF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82Ce
// SIG // RBCToR1E7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIz
// SIG // qVo3A0iMHBsbioxa5DYyXuEfNCktosUlFXfegd514ZJn
// SIG // s6YDr4PlahC0+lK3hgJy5vkCAwEAAaOCAWkwggFlMAkG
// SIG // A1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQM
// SIG // MAoGCCsGAQUFBwMDMGEGA1UdIARaMFgwVgYGZ4EMAQQB
// SIG // MEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5zeW1jYi5j
// SIG // b20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5z
// SIG // eW1jYi5jb20vcnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL
// SIG // 3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIwIKAeoByGGmh0
// SIG // dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUF
// SIG // BwEBBEswSTAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5
// SIG // bWNkLmNvbTAmBggrBgEFBQcwAoYaaHR0cDovL3JiLnN5
// SIG // bWNiLmNvbS9yYi5jcnQwEQYJYIZIAYb4QgEBBAQDAgQQ
// SIG // MBYGCisGAQQBgjcCARsECDAGAQEAAQH/MA0GCSqGSIb3
// SIG // DQEBCwUAA4IBAQDAO29k596Wq5ympOToYLmaRc7ZeGvY
// SIG // x6j5lNWmwCC9ACLTN+mqXF1msf5KtHRHTpyLFVDH/zCC
// SIG // 3LrRzHSHuflhkKFlhJQsEttZ3rhmKxlEtJ85Id9pA8wm
// SIG // XN+Q5tKIStReWLsZ2eA15G9BEFmPvq5DAFj0h+LNF6hq
// SIG // 87C2bVqim29Kf9wDMp3Ndd7hj07QEVh7CqCKEoAJYvXR
// SIG // BBAdrMTqjCTtYCwDkAaAg5LdEm5w76jCQkR2XzKOTgDl
// SIG // qa9uQIyQBAc2ci6X9OBdGJw0ZM0JDlEkRpr3uODtZIOn
// SIG // UQQchI+k+cwEAK7+vFhfsLBi7Dyt1mbvfINr803cFja3
// SIG // i5JOMIIFRzCCBC+gAwIBAgIQfBs1NUrn23TnQV8Racpr
// SIG // qDANBgkqhkiG9w0BAQsFADCBvTELMAkGA1UEBhMCVVMx
// SIG // FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL
// SIG // ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQL
// SIG // EzEoYykgMjAwOCBWZXJpU2lnbiwgSW5jLiAtIEZvciBh
// SIG // dXRob3JpemVkIHVzZSBvbmx5MTgwNgYDVQQDEy9WZXJp
// SIG // U2lnbiBVbml2ZXJzYWwgUm9vdCBDZXJ0aWZpY2F0aW9u
// SIG // IEF1dGhvcml0eTAeFw0xNDA3MjIwMDAwMDBaFw0yNDA3
// SIG // MjEyMzU5NTlaMIGEMQswCQYDVQQGEwJVUzEdMBsGA1UE
// SIG // ChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsT
// SIG // FlN5bWFudGVjIFRydXN0IE5ldHdvcmsxNTAzBgNVBAMT
// SIG // LFN5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
// SIG // bmluZyBDQSAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEA15VD1NzfZ645+1KktiYxBHDpt45b
// SIG // Kro3aTWVj7vAMOeG2HO73+vRdj+KVo7rLUvwVxhOsY2l
// SIG // M9MLdSPVankn3aPT9w6HZbXerRzx9TW0IlGvIqHBXUuQ
// SIG // f8BZTqudeakC1x5JsTtNh/7CeKu/71KunK8I2TnlmlE+
// SIG // aV8wEE5xY2xY4fAgMxsPdL5byxLh24zEgJRyu/ZFmp7B
// SIG // JQv7oxye2KYJcHHswEdMj33D3hnOPu4Eco4X0//wsgUy
// SIG // GUzTsByf/qV4IEJwQbAmjG8AyDoAEUF6QbCnipEEoJl4
// SIG // 9He082Aq5mxQBLcUYP8NUfSoi4T+IdpcXn31KXlPsER0
// SIG // b21y/wIDAQABo4IBeDCCAXQwLgYIKwYBBQUHAQEEIjAg
// SIG // MB4GCCsGAQUFBzABhhJodHRwOi8vcy5zeW1jZC5jb20w
// SIG // EgYDVR0TAQH/BAgwBgEB/wIBADBmBgNVHSAEXzBdMFsG
// SIG // C2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBz
// SIG // Oi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBka
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMDYGA1UdHwQv
// SIG // MC0wK6ApoCeGJWh0dHA6Ly9zLnN5bWNiLmNvbS91bml2
// SIG // ZXJzYWwtcm9vdC5jcmwwEwYDVR0lBAwwCgYIKwYBBQUH
// SIG // AwMwDgYDVR0PAQH/BAQDAgEGMCkGA1UdEQQiMCCkHjAc
// SIG // MRowGAYDVQQDExFTeW1hbnRlY1BLSS0xLTcyNDAdBgNV
// SIG // HQ4EFgQU1MAGIknrOUvdk+JcobhHdglyA1gwHwYDVR0j
// SIG // BBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkwDQYJKoZI
// SIG // hvcNAQELBQADggEBAH/ryqfqi3ZC6z6OIFQw47e53PpI
// SIG // PhbHD0WVEM0nhqNm8wLtcfiqwlWXkXCD+VJ+Umk8yfHg
// SIG // lEaAGLuh1KRWpvMdAJHVhvNIh+DLxDRoIF60y/kF7Zyv
// SIG // cFMnueg+flGgaXGL3FHtgDolMp9Er25DKNMhdbuX2IuL
// SIG // jP6pBEYEhfcVnEsRjcQsF/7Vbn+a4laS8ZazrS359N/a
// SIG // iZnOsjhEwPdHe8olufoqaDObUHLeqJ/UzSwLNL2LMHhA
// SIG // 4I2OJxuQbxq+CBWBXesv4lHnUR7JeCnnHmW/OO8BSgEJ
// SIG // JA4WxBR5wUE3NNA9kVKUneFo7wjw4mmcZ26QCxqTcdQm
// SIG // AsPAWiMxgg7oMIIO5AIBATCBmTCBhDELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1
// SIG // NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMgIQdnK0JhTS6n5k
// SIG // EFOlgziEljANBglghkgBZQMEAgEFAKB8MBAGCisGAQQB
// SIG // gjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MC8GCSqGSIb3DQEJBDEiBCD4HzyfdB5THgpZuAH9FqoE
// SIG // IeS6XUHczM+Ai/rdpMKUgzANBgkqhkiG9w0BAQEFAASC
// SIG // AQAf/8KzTNVTLcKj2i5mFFEY6w4C/FaQUOKUQWO3/B28
// SIG // qUPPQn7HVmY9KkA9P3dpg6dHYZrdF5RgpkMD5hnKpV4+
// SIG // +vz1wmZESD3RSBng+PhDzoDQ7XnJ/hfLMJg+NCshvikH
// SIG // 0aT498kuxDGYet1HuOKllZhKC6DMfiIcgxGBghUcLY3Z
// SIG // uEz6vsoASxPvmmL63eKGg6PHvdLBVOnGOJxfD9geYM5U
// SIG // NAHuhp5e3GdNhovlxlbkxE65PlTHhIexbOi9dqzgBghP
// SIG // cxlqFJN5+aVMgdZvFixmqQvp/CkVPHBldHE9W96neE8Q
// SIG // 3GBIQc1MxkTrdn2XZEtwzyxwHAI8ThAzR7jLoYIMoTCC
// SIG // DJ0GCisGAQQBgjcDAwExggyNMIIMiQYJKoZIhvcNAQcC
// SIG // oIIMejCCDHYCAQMxDzANBglghkgBZQMEAgEFADCB3QYL
// SIG // KoZIhvcNAQkQAQSggc0EgcowgccCAQEGCSsGAQQBoDIC
// SIG // AzAxMA0GCWCGSAFlAwQCAQUABCCQao0ZOhbmSVBhd2RE
// SIG // AhIJoueVeeD/LxcEOHEtC10GvAIUL54M0OnzA4NgjRJu
// SIG // HdurR7f0uR4YDzIwMTcwMjAzMDQ1MDM1WqBdpFswWTEL
// SIG // MAkGA1UEBhMCU0cxHzAdBgNVBAoTFkdNTyBHbG9iYWxT
// SIG // aWduIFB0ZSBMdGQxKTAnBgNVBAMTIEdsb2JhbFNpZ24g
// SIG // VFNBIGZvciBBZHZhbmNlZCAtIEcyoIIIxjCCBKkwggOR
// SIG // oAMCAQICEhEhBvEPzmjwm/rlWxjNjyABdzANBgkqhkiG
// SIG // 9w0BAQsFADBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
// SIG // R2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFs
// SIG // U2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBH
// SIG // MjAeFw0xNjA1MjQwMDAwMDBaFw0yNzA2MjQwMDAwMDBa
// SIG // MFkxCzAJBgNVBAYTAlNHMR8wHQYDVQQKExZHTU8gR2xv
// SIG // YmFsU2lnbiBQdGUgTHRkMSkwJwYDVQQDEyBHbG9iYWxT
// SIG // aWduIFRTQSBmb3IgQWR2YW5jZWQgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBALfHkooo2POR
// SIG // y1ANXespRMGCWaXKZM69g7VR5ZTMboCaF2zc/2LmNkNe
// SIG // AcIMZI3Kd572XXdFuV7IJOtBNxFmN6zIzXSbzLPvTOJ/
// SIG // G85zvsmWnTUefPdU92zsoBLWrpmdY8R4X1mpLiL1wyfY
// SIG // sltFYyeQ/4yxPam08w7A8SBlBomdAxyjsFJBhTTrvMvO
// SIG // VPYS/rMBiUqm+lTFH/vTHMDjv5fjP9Ab+UDHG9XrJnxD
// SIG // MMdw8ngRqoVOpQ4NAEo6EXejyiMBgJ7Ik1ZdRsyK2NKq
// SIG // CoSFsolb1TLOQXsYTlTKq9FSXhLTJJ5W8wyP3b2SjnnV
// SIG // QYnDo6DlkfzHZ52HM85xMnMCAwEAAaOCAWcwggFjMA4G
// SIG // A1UdDwEB/wQEAwIHgDBMBgNVHSAERTBDMEEGCSsGAQQB
// SIG // oDIBHjA0MDIGCCsGAQUFBwIBFiZodHRwczovL3d3dy5n
// SIG // bG9iYWxzaWduLmNvbS9yZXBvc2l0b3J5LzAJBgNVHRME
// SIG // AjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMEYGA1Ud
// SIG // HwQ/MD0wO6A5oDeGNWh0dHA6Ly9jcmwuZ2xvYmFsc2ln
// SIG // bi5jb20vZ3MvZ3N0aW1lc3RhbXBpbmdzaGEyZzIuY3Js
// SIG // MFgGCCsGAQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0
// SIG // cDovL3NlY3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQv
// SIG // Z3N0aW1lc3RhbXBpbmdzaGEyZzIuY3J0MB0GA1UdDgQW
// SIG // BBQtbm7RjeUDgO7nY+mn2doLPFciPTAfBgNVHSMEGDAW
// SIG // gBSSIadKlV1ksJu0HuYAN0fmnUErTDANBgkqhkiG9w0B
// SIG // AQsFAAOCAQEAV51T5N3upSze5L9igKJhhkqfm50kIzCb
// SIG // jyeHL/oEWc5wiD1GUnfEm0XSj723IRhJ2C6H/5Iud/k/
// SIG // CvmgIVwTT+SEKyiHzFwVuROr4hJVw/hFHkkZzqp1DyHo
// SIG // 71H8NCwLMgUJsuQWaa3ZLn7h/C1IvxrTdDUBOt8wQ3Bn
// SIG // ejjXuhHCVvsxSLpb8SESYuB2iZEfSTjUWE15CYqp2m8C
// SIG // 1q3k2ol9TNmxMHBAattFulN2kNxLQhYhz+TSWJTUVWWb
// SIG // dgOsrhgItoMSjEE+X4BFZMiJ1DMXoaFMvT/Ekv5/hfK+
// SIG // sazX9p7LzhMq7gJDT/z/cDU0ozN8z+INMiQgfAA+ozIR
// SIG // 3jCCBBUwggL9oAMCAQICCwQAAAAAATGJxlAEMA0GCSqG
// SIG // SIb3DQEBCwUAMEwxIDAeBgNVBAsTF0dsb2JhbFNpZ24g
// SIG // Um9vdCBDQSAtIFIzMRMwEQYDVQQKEwpHbG9iYWxTaWdu
// SIG // MRMwEQYDVQQDEwpHbG9iYWxTaWduMB4XDTExMDgwMjEw
// SIG // MDAwMFoXDTI5MDMyOTEwMDAwMFowWzELMAkGA1UEBhMC
// SIG // QkUxGTAXBgNVBAoTEEdsb2JhbFNpZ24gbnYtc2ExMTAv
// SIG // BgNVBAMTKEdsb2JhbFNpZ24gVGltZXN0YW1waW5nIENB
// SIG // IC0gU0hBMjU2IC0gRzIwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCqm47DqxFRJQG2lpTiT9jBCPZG
// SIG // I9lFxZWXW6sav9JsV8kzBh+gD8Y8flNIer+dh56v7sOM
// SIG // R+FC7OPjoUpsDBfEpsG5zVvxHkSJjv4L3iFYE+5NyMVn
// SIG // Cxyys/E0dpGiywdtN8WgRyYCFaSQkal5ntfrV50rfCLY
// SIG // FNfxBx54IjZrd3mvr/l/jk7htQgx/ertS3FijCPxAzmP
// SIG // RHm2dgNXnq0vCEbc0oy89I50zshoaVF2EYsPXSRbGVQ9
// SIG // JsxAjYInG1kgfVn2k4CO+Co4/WugQGUfV3bMW44ETyyo
// SIG // 24RQE0/G3Iu5+N1pTIjrnHswJvx6WLtZvBRykoFXt3bJ
// SIG // 2IAKgG4JAgMBAAGjgegwgeUwDgYDVR0PAQH/BAQDAgEG
// SIG // MBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFJIh
// SIG // p0qVXWSwm7Qe5gA3R+adQStMMEcGA1UdIARAMD4wPAYE
// SIG // VR0gADA0MDIGCCsGAQUFBwIBFiZodHRwczovL3d3dy5n
// SIG // bG9iYWxzaWduLmNvbS9yZXBvc2l0b3J5LzA2BgNVHR8E
// SIG // LzAtMCugKaAnhiVodHRwOi8vY3JsLmdsb2JhbHNpZ24u
// SIG // bmV0L3Jvb3QtcjMuY3JsMB8GA1UdIwQYMBaAFI/wS3+o
// SIG // LkUkrk1Q+mOai97i3Ru8MA0GCSqGSIb3DQEBCwUAA4IB
// SIG // AQAEVoJKfNDOyb82ZtG+NZ6TbJfoBs4xGFn5bEFfgC7A
// SIG // QiW4GMf81LE3xGigzyhqA3RLY5eFd2E71y/j9b0zopJ9
// SIG // ER+eimzvLLD0Yo02c9EWNvG8Xuy0gJh4/NJ2eejhIZTg
// SIG // H8Si4apn27Occ+VAIs85ztvmd5Wnu7LL9hmGnZ/I1JgF
// SIG // snFvTnWu8T1kajteTkamKl0IkvGj8x10v2INI4xcKjiV
// SIG // 0sDVzc+I2h8otbqBaWQqtaai1XOv3EbbBK6R127FmLrU
// SIG // R8RWdIBHeFiMvu8r/exsv9GU979Q4HvgkP0gGHgYIl0I
// SIG // LowcoJfzHZl9o52R0wZETgRuehwg4zbwtlC5MYICtDCC
// SIG // ArACAQEwcTBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
// SIG // R2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFs
// SIG // U2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBH
// SIG // MgISESEG8Q/OaPCb+uVbGM2PIAF3MA0GCWCGSAFlAwQC
// SIG // AQUAoIIBFDAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQ
// SIG // AQQwHAYJKoZIhvcNAQkFMQ8XDTE3MDIwMzA0NTAzNVow
// SIG // LwYJKoZIhvcNAQkEMSIEIB4vSZ/QaZY3N5xutmNVbBjq
// SIG // 4iBHCLBxYGxWq2JTFBBeMIGmBgsqhkiG9w0BCRACDDGB
// SIG // ljCBkzCBkDCBjQQUfVXY51pWovxzgkP3uFSHXFy1Kg0w
// SIG // dTBfpF0wWzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEds
// SIG // b2JhbFNpZ24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNp
// SIG // Z24gVGltZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzIC
// SIG // EhEhBvEPzmjwm/rlWxjNjyABdzANBgkqhkiG9w0BAQEF
// SIG // AASCAQCtDwW84Q0drx+KmTkXpEs36RGgOvEEWozH40bj
// SIG // UirDoS4Ko/bid0UvbtveoWZwJh5s6jyPu25lobocj6d5
// SIG // iDO/N8a/sfbCF7HPAvrvwN0od1l0vF2mLEXYfyzvxmU8
// SIG // GnQA0qch9DCxBqGAnbW6lFTCGWQeQb1LVA4e0siS+u95
// SIG // YrbHfKqDMO8fDyoMMOYhJx38s+GI9yvMXkzTW0dXFnxM
// SIG // eXWini3A1T+LRwux11ML3zR542Dv7oK+drhkIMXu+LcW
// SIG // DAL+CWOZ8PIqhBWM2gZYtU/bSONz+bXYWn7bZP+FqZPW
// SIG // 7XS/vpTIyCDlzqfCnThT7qOBpXgYnYkuM0yf2lJJ
// SIG // End signature block
